;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2015 Sou Bunnbu <iyzsong@gmail.com>
;;; Copyright © 2015 Mark H Weaver <mhw@netris.org>
;;; Copyright © 2016 Efraim Flashner <efraim@flashner.co.il>
;;; Copyright © 2018 Tobias Geerinckx-Rice <me@tobias.gr>
;;; Copyright © 2020 Marius Bakke <mbakke@fastmail.com>
;;; Copyright © 2025 Maxim Cournoyer <maxim@guixotic.coop>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages ninja)
  #:use-module ((guix licenses) #:select (asl2.0 expat))
  #:use-module (guix gexp)
  #:use-module (guix packages)
  #:use-module (guix utils)
  #:use-module (guix git-download)
  #:use-module (guix build-system gnu)
  #:use-module (guix build-system trivial)
  #:use-module (guix utils)
  #:use-module (gnu packages)
  #:use-module (gnu packages bash)
  #:use-module (gnu packages python)
  #:use-module (gnu packages re2c)
  #:use-module (srfi srfi-1))

(define-public ninja/pinned
  (hidden-package
   (package
     (name "ninja")
     (version "1.13.1")
     (source (origin
               (method git-fetch)
               (uri (git-reference
                     (url "https://github.com/ninja-build/ninja")
                     (commit (string-append "v" version))))
               (file-name (git-file-name name version))
               (sha256
                (base32
                 "0vil4mz0h1z39d2airzdi8cia8xhn3n5p94pv4sd3mqk0pkha40s"))))
     (build-system gnu-build-system)
     (inputs (list python-wrapper re2c))
     (arguments
      '(; Tests now require googletest, which is a circular dependency.
        #:tests? #f
        #:phases
        (modify-phases %standard-phases
          (replace 'configure
            (lambda _
              (substitute* "src/subprocess-posix.cc"
                (("/bin/sh") (which "sh")))
              (substitute* "src/subprocess_test.cc"
                (("/bin/echo") (which "echo")))))
          (replace 'build
            (lambda _
              (invoke "./configure.py" "--bootstrap")))
          (replace 'check
            (lambda* (#:key tests? #:allow-other-keys)
              (when tests?
                (invoke "./configure.py")
                (invoke "./ninja" "ninja_test")
                (invoke "./ninja_test"))))
          (replace 'install
            (lambda* (#:key outputs #:allow-other-keys)
              (let* ((out (assoc-ref outputs "out"))
                     (bin (string-append out "/bin"))
                     (doc (string-append out "/share/doc/ninja")))
                (install-file "ninja" bin)
                (install-file "doc/manual.asciidoc" doc)))))))
     (home-page "https://ninja-build.org/")
     (synopsis "Small build system")
     (description
      "Ninja is a small build system with a focus on speed.  It differs from
other build systems in two major respects: it is designed to have its input
files generated by a higher-level build system, and it is designed to run
builds as fast as possible.")
    (license asl2.0))))

(define-public ninja
  (package/inherit ninja/pinned
    (properties (alist-delete 'hidden? (package-properties ninja/pinned)))))

(define-public samurai
  (package
    (name "samurai")
    (version "1.2")
    (source (origin
              (method git-fetch)
              (uri (git-reference
                    (url "https://github.com/michaelforney/samurai")
                    (commit version)))
              (file-name (git-file-name name version))
              (sha256
               (base32
                "04iw18hgkm72yrl83a2xh1jc47w9rilpb95kwick0j37b4q3gxj4"))))
    (build-system gnu-build-system)
    (arguments
     (list #:tests? #f                  ;no test suite
           #:make-flags #~(list (string-append "CC=" #$(cc-for-target))
                                (string-append "PREFIX=" #$output))
           #:phases #~(modify-phases %standard-phases
                        (add-after 'unpack 'patch-/bin/sh
                          (lambda* (#:key inputs #:allow-other-keys)
                            (substitute* "build.c"
                              (("/bin/sh")
                               (search-input-file inputs "bin/sh")))))
                        (delete 'configure))))
    (inputs (list bash-minimal))
    (home-page "https://github.com/michaelforney/samurai")
    (synopsis "Ninja-compatible build tool written in C")
    (description "@code{samurai} is a ninja-compatible build tool written in
C99 with a focus on simplicity, speed, and portability.  It is
feature-complete and supports most of the same options as ninja.  This package
provides the command @command{samu}.  For a @command{ninja} command
replacement, use the @code{samu-as-ninja-wrapper} package.")
    (license (list asl2.0               ;for the most part
                   expat))))            ;htab.h, htab.c and tree.c

(define-public samu-as-ninja-wrapper
  (package/inherit samurai
    (name "samu-as-ninja-wrapper")
    (build-system trivial-build-system)
    (arguments
     (list #:builder
           (with-imported-modules '((guix build utils))
             #~(begin
                 (use-modules (guix build utils))
                 (let ((bindir (string-append #$output "/bin"))
                       (samu (string-append #$(this-package-input "samurai")
                                            "/bin/samu")))
                   (mkdir-p bindir)
                   (symlink samu (string-append bindir "/ninja")))))))
    (inputs (list samurai))
    (description "This package provides the @command{ninja} command,
implemented as a symbolic link to the @command{samu} command of @code{samurai}
package.")))
