;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2017 Ricardo Wurmus <rekado@elephly.net>
;;; Copyright © 2021-2022 Ludovic Courtès <ludo@gnu.org>
;;; Copyright © 2021 Thiago Jung Bauermann <bauermann@kolabnow.com>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (guix build-system texlive)
  #:use-module (guix store)
  #:use-module (guix utils)
  #:use-module (guix packages)
  #:use-module (guix monads)
  #:use-module (guix gexp)
  #:use-module (guix search-paths)
  #:use-module (guix build-system)
  #:use-module (guix build-system gnu)
  #:use-module (guix svn-download)
  #:export (%texlive-build-system-modules
            %texlive-repository
            texlive-build
            texlive-build-system
            texlive-packages-repository))

;; Commentary:
;;
;; Standard build procedure for Texlive packages.
;;
;; Code:

(define %texlive-repository "svn://www.tug.org/texlive/")

(define (texlive-packages-repository version)
  "Return URL for packages location in TeX Live repository, at VERSION."
  (string-append
   %texlive-repository "tags/texlive-" version "/Master/texmf-dist"))

(define %texlive-build-system-modules
  ;; Build-side modules imported by default.
  `((guix build texlive-build-system)
    (guix build union)
    ,@%default-gnu-imported-modules))

(define (default-texlive-bin)
  "Return the default texlive-bin package, resolved lazily."
  (@* (gnu packages tex) texlive-bin))

(define (texlive-latex-bin)
  "Return the default texlive-latex-bin package, resolved lazily."
  (@* (gnu packages tex) texlive-latex-bin))

(define* (lower name
                #:key
                source inputs native-inputs outputs
                system target
                (texlive-latex-bin? #true)
                (texlive-bin (default-texlive-bin))
                #:allow-other-keys
                #:rest arguments)
  "Return a bag for NAME."
  (define private-keywords
    '(#:target #:inputs #:native-inputs
      #:texlive-latex-bin? #:texlive-bin))

  (bag
    (name name)
    (system system)
    (host-inputs `(,@(if source
                         `(("source" ,source))
                         '())
                   ,@inputs

                   ;; Keep the standard inputs of 'gnu-build-system'.
                   ,@(standard-packages)))
    (build-inputs `(("texlive-bin" ,texlive-bin)
                    ,@(if texlive-latex-bin?
                          `(("texlive-latex-bin" ,(texlive-latex-bin)))
                          '())
                    ,@native-inputs))
    (outputs outputs)
    (build texlive-build)
    (arguments (strip-keyword-arguments private-keywords arguments))))

(define* (texlive-build name inputs
                        #:key
                        source
                        (tests? #f)
                        (build-targets #f)
                        (create-formats #f)
                        (link-scripts #f)
                        (tex-engine #f)

                        ;; FIXME: This would normally default to "luatex" but
                        ;; LuaTeX has a bug where sometimes it corrupts the
                        ;; heap and aborts. This causes the build of texlive
                        ;; packages to fail at random. The problem is being
                        ;; tracked at <https://issues.guix.gnu.org/48064>.
                        (tex-format "pdftex")

                        (phases '(@ (guix build texlive-build-system)
                                    %standard-phases))
                        (outputs '("out"))
                        (search-paths '())
                        (system (%current-system))
                        (guile #f)
                        (substitutable? #t)
                        (imported-modules %texlive-build-system-modules)
                        (modules '((guix build texlive-build-system)
                                   (guix build union)
                                   (guix build utils))))
  "Build SOURCE with INPUTS."
  (define builder
    (with-imported-modules imported-modules
      #~(begin
          (use-modules #$@(sexp->gexp modules))

          #$(with-build-variables inputs outputs
              #~(texlive-build #:name #$name
                               #:source #+source
                               #:build-targets #$build-targets
                               #:create-formats #$create-formats
                               #:link-scripts #$link-scripts
                               #:tex-engine #$(if tex-engine
                                                  tex-engine
                                                  tex-format)
                               #:tex-format #$tex-format
                               #:system #$system
                               #:tests? #$tests?
                               #:phases #$(if (pair? phases)
                                              (sexp->gexp phases)
                                              phases)
                               #:outputs %outputs
                               #:inputs %build-inputs
                               #:search-paths '#$(sexp->gexp
                                                  (map search-path-specification->sexp
                                                       search-paths)))))))

  (mlet %store-monad ((guile (package->derivation (or guile (default-guile))
                                                  system #:graft? #f)))
    (gexp->derivation name builder
                      #:system system
                      #:target #f
                      #:graft? #f
                      #:substitutable? substitutable?
                      #:guile-for-build guile)))

(define texlive-build-system
  (build-system
    (name 'texlive)
    (description "The build system for TeX Live packages")
    (lower lower)))

;;; texlive.scm ends here
